#! /usr/bin/perl

###########################################################################

use strict;
use FileHandle;
use Getopt::Long;
use Pod::Text;

###########################################################################

my $endl  = "\r\n";
my $debug = undef;

###########################################################################

package Util;

sub is_element {
    my $class = shift;
    my $name  = shift;

    return $name eq "tC~[Y"
        || $name eq "ANAY"
        || $name eq "GAX"
        || $name eq "A[V[Y";
}

###########################################################################

package Devil;

sub new {
    my $class    = shift;
    my $category = shift;
    my $name     = shift;
    my $level    = shift;

    my $this = {
        category => $category,
        name     => $name,
        level    => $level,
    };
    bless $this, $class;

    return $this;
}

sub more_equal {
    my $this  = shift;
    my $level = shift;

    return $this->{level} >= $level;
}

###########################################################################

package DevilTable;

sub new {
    my $class    = shift;
    my $filepath = shift;

    my $this = {
        category_table => {},
        name_table     => {},
        list           => [],
    };
    bless $this, $class;

    $this->load($filepath);

    return $this;
}

sub load {
    my $this     = shift;
    my $filepath = shift;

    my $h = FileHandle->new($filepath);
    $h or die "Couldn't open $filepath: $!";
    binmode $h;

    $h->getline; # title
    while (my $buffer = $h->getline) {

        $buffer =~ s/$endl//;
        my($category, $name, $level) = split /\t/, $buffer;

        $debug and print "[DevilTable::load] $category, $name, $level\n";
        my $devil = new Devil($category, $name, $level);

        exists $this->{category_table}{$category} or $this->{category_table}{$category} = [];
        push @{$this->{category_table}{$category}}, $devil;
        $this->{name_table}{$name} = $devil;
        push @{$this->{list}}, $devil;
    }
}

sub find_by_category_and_level {
    my $this     = shift;
    my $category = shift;
    my $level    = shift;

    exists $this->{category_table}{$category} or die "Unknown category: $category";

    my @devils = reverse @{$this->{category_table}{$category}};
    for my $devil (@devils) {
        $devil->more_equal($level) and return $devil;
    }

    return undef;
}

sub find_by_name {
    my $this = shift;
    my $name = shift;

    exists $this->{name_table}{$name} or die "Unknown name: $name";
    return $this->{name_table}{$name};
}

sub list {
    my $this = shift;
    return $this->{list};
}

###########################################################################

package PriorityTable;

sub new {
    my $class    = shift;
    my $filepath = shift;

    my $this = {
        table => {},
    };
    bless $this, $class;

    $this->load($filepath);

    return $this;
}

sub load {
    my $this     = shift;
    my $filepath = shift;

    my $h = FileHandle->new($filepath);
    $h or die "Couldn't open $filepath: $!";
    binmode $h;

    $h->getline; # title
    while (my $buffer = $h->getline) {
        $buffer =~ s/$endl//;
        my($priority, $category) = split /\t/, $buffer;

        $debug and print "[PriorityTable::load] $priority, $category\n";
        $this->{table}{$category} = $priority;
    }
}

sub compare {
    my $this    = shift;
    my $devil_a = shift;
    my $devil_b = shift;
    my $category_a = $devil_a->{category};
    my $category_b = $devil_b->{category};
    return $this->{table}{$category_a} <=> $this->{table}{$category_b};
}

###########################################################################

package RemixTable;

sub new {
    my $class = shift;

    my $this = {
        table => {},
    };
    bless $this, $class;

    return $this;
}

sub load {
    my $this     = shift;
    my $filepath = shift;
    my $option   = shift; # enable_sharped, disable_sharped

    my $h = FileHandle->new($filepath);
    $h or die "Couldn't open $filepath: $!";
    binmode $h;

    my $buffer = $h->getline;
    $buffer =~ s/^.*?\t//;
    $buffer =~ s/$endl//;

    my @category = split /\t/, $buffer;
    $debug and print "[RemixTable::load] @category\n";

    foreach my $category_row (@category) {
        my $buffer = $h->getline;
        $buffer =~ s/^.*?\t//;
        $buffer =~ s/$endl//;
        my @result = split /\t/, $buffer;

        for (my $i = 0; $i < @category; ++$i) {
            my $category_col = $category[$i];
            my $result       = $result  [$i];

            $result eq "" and next;
            Util->is_element($result) and next;

            if ($result =~ s/^#//) {
                $option =~ /disable_sharped/ and next;
            }

            my $key = "$category_col:$category_row";
            $debug and print "[RemixTable::load] $key, $result\n";
            $this->{table}{$key} = $result;
        }
    }
}

sub remix {
    my $this    = shift;
    my $devil_a = shift;
    my $devil_b = shift;
    my $key     = "$devil_a->{category}:$devil_b->{category}";

    return $this->{table}{$key};
}

###########################################################################

package Result;

sub new {
    my $class       = shift;
    my $devil_table = shift;

    my $this = {
        table    => {},
        list_ref => $devil_table->list,
    };
    bless $this, $class;

    %{$this->{table}} = map { $_->{name}, [] } @{$this->{list_ref}};

    return $this;
}

sub add {
    my $this    = shift;
    my $remixed = shift;
    my @devil   = @_;

    my $remixed_name = $remixed->{name};
    my @devil_name   = map { $_->{name} } @devil;

    push @{$this->{table}{$remixed_name}}, \@devil_name;
}

sub generate_id {
    my $this = shift;
    my $id   = shift;

    $id !~  /^[a-zA-Z]/ and $id = "ID$id";
    $id =~ s|([^a-zA-Z0-9.\-_])|sprintf(":%2.2X", unpack("C", $1))|eg;

    return $id;
}

sub save_as_text {
    my $this     = shift;
    my $filepath = shift;

    my $h = FileHandle->new("> $filepath");
    $h or die "Couldn't open $filepath: $!";
    binmode $h;

    foreach my $remixed (@{$this->{list_ref}}) {

        $h->print("$remixed->{name} ($remixed->{category} Lv.$remixed->{level})\n");

        my $devil_name_list = $this->{table}{$remixed->{name}};
        foreach my $devil_name (@$devil_name_list) {
            $h->print("\t", join(" + ", @$devil_name), "\n");
      }
    }
}

sub save_as_html {
    my $this     = shift;
    my $filepath = shift;

    my $h = FileHandle->new("> $filepath");
    $h or die "Couldn't open $filepath: $!";
    binmode $h;

    foreach my $remixed (@{$this->{list_ref}}) {

        my $id = $this->generate_id($remixed->{name});

        $h->print(<< "END_OF_HERE");
<dt><a id="$id">$remixed->{name}</a> ($remixed->{category} Lv.$remixed->{level})</dt>
END_OF_HERE

        my $devil_name_list = $this->{table}{$remixed->{name}};
        if (@$devil_name_list) {
            $h->print("<dd><ul>\n");
            foreach my $devil_name (@$devil_name_list) {
                $h->print("    <li>", join(" + ", map {
                    qq|<a href="#| . $this->generate_id($_) . qq|">$_</a>|
                } @$devil_name), "</li>\n");
            }
            $h->print("</ul></dd>\n");
        } else {
            $h->print("<dd><p>nil</p></dd>\n");
        }
    }
}

###########################################################################

package main;

#--------------------------------------------------------------------------

sub help() {
    pod2text $0;
    exit;
}

my %optctl;
GetOptions \%optctl, "help!", "output=s" or help;
$optctl{help} and help;

#--------------------------------------------------------------------------

my $devil_table    = DevilTable->new("data/devil.txt");
my $priority_table = PriorityTable->new("data/priority.txt");

my $remix2_table = RemixTable->new;
$remix2_table->load("data/ln.txt", "disable_sharped");
$remix2_table->load("data/dn.txt");

my $remix3_table = RemixTable->new;
$remix3_table->load("data/ln.txt", "enable_sharped");
$remix3_table->load("data/dn.txt");

my $result = Result->new($devil_table);

#--------------------------------------------------------------------------

sub remix2($$) {
    my $devil_a = shift;
    my $devil_b = shift;

    my $category = $remix2_table->remix($devil_a, $devil_b);
    defined $category or return undef;

    my $level = ($devil_a->{level} + $devil_b->{level}) / 2 + 3;
    return $devil_table->find_by_category_and_level($category, $level);
}

sub remix3($$$) {
    my $devil_a = shift;
    my $devil_b = shift;
    my $devil_c = shift;

    my $tmp_devil = remix2($devil_b, $devil_c);
    defined $tmp_devil or return undef;

    my @devil    = sort { $priority_table->compare($a, $b) } ($devil_a, $tmp_devil);
    my $category = $remix3_table->remix($devil[0], $devil[1]);
    defined $category or return undef;

    my $level = ($devil_a->{level} + $devil_b->{level} + $devil_c->{level}) / 3 + 3;
    return $devil_table->find_by_category_and_level($category, $level);
}

#--------------------------------------------------------------------------

sub remix_all() {

    foreach my $devil_a (@{$devil_table->list}) {
        foreach my $devil_b (@{$devil_table->list}) {
            foreach my $devil_c (@{$devil_table->list}) {
                $priority_table->compare($devil_a, $devil_b) == -1 or next;
                $priority_table->compare($devil_b, $devil_c) == -1 or next;
                my $remixed = remix3($devil_a, $devil_b, $devil_c);
                defined $remixed and $result->add($remixed, $devil_a, $devil_b, $devil_c);
            }
        }
    }

    foreach my $devil_a (@{$devil_table->list}) {
        foreach my $devil_b (@{$devil_table->list}) {
            $priority_table->compare($devil_a, $devil_b) == -1 or next;
            my $remixed = remix2($devil_a, $devil_b);
            defined $remixed and $result->add($remixed, $devil_a, $devil_b);
        }
    }
}

#--------------------------------------------------------------------------

remix_all();

my $output_filepath = exists $optctl{output}
  ? $optctl{output}
  : "nine_revmap.txt";
$result->save_as_text($output_filepath);

###########################################################################

=head1 NAME

revmap.pl - helper script for DDS NINE

=head1 SYNOPSYS

    $ ./revmap.pl
    $ ./revmap.pl --output=nine_revmap.txt

=head1 ARGUMENTS

=over 4

=item --help

show this message

=item --output=FILEPATH

specify the destination file

=back

=cut

###########################################################################
